
import { useState, useEffect, memo } from "react";
import { cn } from "@/lib/utils";

interface OptimizedImageProps {
  src: string;
  alt: string;
  className?: string;
  width?: number;
  height?: number;
  loading?: "lazy" | "eager";
  onClick?: () => void;
}

const OptimizedImage = ({
  src,
  alt,
  className,
  width,
  height,
  loading = "lazy",
  onClick,
}: OptimizedImageProps) => {
  const [isLoaded, setIsLoaded] = useState(false);
  const [error, setError] = useState(false);

  // Reset states if source changes
  useEffect(() => {
    setIsLoaded(false);
    setError(false);
  }, [src]);

  const handleImageLoad = () => {
    setIsLoaded(true);
  };

  const handleImageError = () => {
    setError(true);
    console.error(`Failed to load image: ${src}`);
  };

  return (
    <div className={cn("relative overflow-hidden", !isLoaded && "bg-slate-100", className)}>
      {!isLoaded && !error && (
        <div className="absolute inset-0 flex items-center justify-center bg-slate-100">
          <div className="h-8 w-8 animate-pulse rounded-full bg-slate-200"></div>
        </div>
      )}
      
      {error ? (
        <div className="flex h-full w-full items-center justify-center bg-slate-100 p-4">
          <span className="text-sm text-slate-500">Image non disponible</span>
        </div>
      ) : (
        <img
          src={src}
          alt={alt}
          className={cn(
            "h-full w-full object-cover transition-opacity duration-300",
            isLoaded ? "opacity-100" : "opacity-0"
          )}
          loading={loading}
          width={width}
          height={height}
          onLoad={handleImageLoad}
          onError={handleImageError}
          onClick={onClick}
          decoding="async"
          fetchPriority={loading === "eager" ? "high" : "auto"}
        />
      )}
    </div>
  );
};

// Utilisation de memo pour éviter les re-renders inutiles
export default memo(OptimizedImage);
