
import { useState } from "react";
import { toast } from "sonner";
import { Send } from "lucide-react";
import { useForm } from "react-hook-form";
import { z } from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";

// Schéma de validation
const formSchema = z.object({
  name: z.string().min(2, { message: "Le nom doit contenir au moins 2 caractères" }),
  email: z.string().email({ message: "Veuillez entrer une adresse email valide" }),
  subject: z.string().min(1, { message: "Veuillez sélectionner un sujet" }),
  message: z.string().min(10, { message: "Le message doit contenir au moins 10 caractères" }),
});

type FormValues = z.infer<typeof formSchema>;

const ContactForm = () => {
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [dialogMessage, setDialogMessage] = useState("");

  const form = useForm<FormValues>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      name: "",
      email: "",
      subject: "",
      message: "",
    },
  });

  const { register, handleSubmit, formState: { errors }, reset } = form;

  const onSubmit = (data: FormValues) => {
    setIsSubmitting(true);

    // Simulation d'envoi du formulaire (à remplacer par l'API réelle)
    setTimeout(() => {
      setIsSubmitting(false);
      toast.success("Votre message a été envoyé avec succès !");
      
      // Préparation du message pour le dialogue de confirmation
      setDialogMessage(`Nous avons bien reçu votre message, ${data.name}. Notre équipe vous répondra dans les plus brefs délais à l'adresse ${data.email}.`);
      setIsDialogOpen(true);
      
      // Réinitialisation du formulaire
      reset();
    }, 1500);
  };

  return (
    <div className="lg:col-span-2 bg-white p-6 rounded-xl shadow-sm border border-ecofor-medium-blue/30 animate-slide-up delay-200">
      <h3 className="text-xl font-semibold mb-4">Envoyez-nous un message</h3>
      
      <form onSubmit={handleSubmit(onSubmit)} className="space-y-5">
        <div className="grid grid-cols-1 sm:grid-cols-2 gap-5">
          <div className="space-y-2">
            <Label htmlFor="name" className="text-sm font-medium">Nom</Label>
            <Input
              id="name"
              placeholder="Votre nom"
              {...register("name")}
              className="w-full p-3 rounded-md border border-ecofor-medium-blue/40 focus:outline-none focus:ring-2 focus:ring-ecofor-dark-green focus:border-transparent"
            />
            {errors.name && (
              <p className="text-sm font-medium text-destructive">{errors.name.message}</p>
            )}
          </div>
          
          <div className="space-y-2">
            <Label htmlFor="email" className="text-sm font-medium">Email</Label>
            <Input
              id="email"
              type="email"
              placeholder="votre.email@exemple.com"
              {...register("email")}
              className="w-full p-3 rounded-md border border-ecofor-medium-blue/40 focus:outline-none focus:ring-2 focus:ring-ecofor-dark-green focus:border-transparent"
            />
            {errors.email && (
              <p className="text-sm font-medium text-destructive">{errors.email.message}</p>
            )}
          </div>
        </div>
        
        <div className="space-y-2">
          <Label htmlFor="subject" className="text-sm font-medium">Sujet</Label>
          <select
            id="subject"
            {...register("subject")}
            className="w-full p-3 rounded-md border border-ecofor-medium-blue/40 focus:outline-none focus:ring-2 focus:ring-ecofor-dark-green focus:border-transparent"
          >
            <option value="">Sélectionnez un sujet</option>
            <option value="Information">Demande d'information</option>
            <option value="Collaboration">Proposition de collaboration</option>
            <option value="Events">Événements</option>
            <option value="Other">Autre</option>
          </select>
          {errors.subject && (
            <p className="text-sm font-medium text-destructive">{errors.subject.message}</p>
          )}
        </div>
        
        <div className="space-y-2">
          <Label htmlFor="message" className="text-sm font-medium">Message</Label>
          <Textarea
            id="message"
            {...register("message")}
            rows={5}
            placeholder="Votre message..."
            className="w-full p-3 rounded-md border border-ecofor-medium-blue/40 focus:outline-none focus:ring-2 focus:ring-ecofor-dark-green focus:border-transparent"
          />
          {errors.message && (
            <p className="text-sm font-medium text-destructive">{errors.message.message}</p>
          )}
        </div>
        
        <div>
          <Button 
            type="submit" 
            className="w-full sm:w-auto bg-ecofor-dark-green hover:bg-ecofor-dark-green/90" 
            size="lg"
            disabled={isSubmitting}
          >
            {isSubmitting ? (
              <span className="flex items-center">
                <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                  <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                  <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                </svg>
                Envoi en cours...
              </span>
            ) : (
              <span className="flex items-center">
                <Send className="mr-2 h-4 w-4" />
                Envoyer le message
              </span>
            )}
          </Button>
        </div>
      </form>
      
      {/* Dialog de confirmation */}
      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="sm:max-w-md">
          <DialogHeader>
            <DialogTitle>Message envoyé</DialogTitle>
          </DialogHeader>
          <div className="space-y-3">
            <p>{dialogMessage}</p>
            <Button
              onClick={() => setIsDialogOpen(false)}
              className="w-full bg-ecofor-dark-green hover:bg-ecofor-dark-green/90"
            >
              Fermer
            </Button>
          </div>
        </DialogContent>
      </Dialog>
    </div>
  );
};

export default ContactForm;
