
import { useState } from "react";
import { Link, useLocation } from "react-router-dom";
import { ChevronDown, ChevronUp } from "lucide-react";
import { cn } from "@/lib/utils";
import { navStructure } from "./navStructure";

interface MobileNavigationProps {
  isOpen: boolean;
}

const MobileNavigation = ({ isOpen }: MobileNavigationProps) => {
  const location = useLocation();
  const [openMenus, setOpenMenus] = useState<string[]>([]);
  const [openSubMenus, setOpenSubMenus] = useState<string[]>([]);

  // Déterminer si un menu ou sous-menu est actif
  const isActive = (href: string, dropdownItems: Array<any> = []) => {
    if (location.pathname === href) return true;
    if (dropdownItems?.length) {
      for (const item of dropdownItems) {
        if (location.pathname === item.href) return true;
        if (item.subDropdownItems?.length) {
          for (const subItem of item.subDropdownItems) {
            if (location.pathname === subItem.href) return true;
          }
        }
      }
    }
    return false;
  };

  const toggleMenu = (menuName: string, e: React.MouseEvent) => {
    e.stopPropagation(); // Empêcher la propagation de l'événement
    setOpenMenus(prevState => 
      prevState.includes(menuName)
        ? prevState.filter(item => item !== menuName)
        : [...prevState, menuName]
    );
  };

  const toggleSubMenu = (menuName: string, e: React.MouseEvent) => {
    e.preventDefault(); // Empêcher la navigation lors du clic sur le bouton
    setOpenSubMenus(prevState => 
      prevState.includes(menuName)
        ? prevState.filter(item => item !== menuName)
        : [...prevState, menuName]
    );
  };

  if (!isOpen) return null;

  return (
    <div className="md:hidden fixed inset-0 top-16 z-50 bg-[hsl(var(--background-secondary))] shadow-lg p-6 animate-fade-in overflow-y-auto max-h-[calc(100vh-4rem)]">
      <nav className="flex flex-col space-y-4 pb-16">
        {navStructure.map((item) => (
          <div key={item.name} className="flex flex-col">
            <div 
              className={cn(
                "flex justify-between items-center py-2 border-b border-forestt-primary/20 transition-colors",
                isActive(item.href, item.dropdownItems)
                  ? "text-forestt-primary border-forestt-primary"
                  : "text-foreground"
              )}
            >
              <Link 
                to={item.href}
                className="text-xl font-medium"
              >
                {item.name}
              </Link>
              
              {item.hasDropdown && (
                <button
                  onClick={(e) => toggleMenu(item.name, e)}
                  className="p-2 rounded-md"
                  aria-expanded={openMenus.includes(item.name)}
                  aria-label={`Afficher le sous-menu de ${item.name}`}
                >
                  {openMenus.includes(item.name) ? (
                    <ChevronUp className="h-5 w-5" />
                  ) : (
                    <ChevronDown className="h-5 w-5" />
                  )}
                </button>
              )}
            </div>

            {/* Sous-menus affichés directement sous le menu parent */}
            {item.hasDropdown && openMenus.includes(item.name) && (
              <div className="pl-4 mt-1 mb-2 space-y-2 border-l-2 border-forestt-primary/20">
                {item.dropdownItems.map((subItem) => (
                  <div key={subItem.name} className="flex flex-col">
                    <div className="flex justify-between items-center py-2">
                      <Link 
                        to={subItem.href}
                        className={cn(
                          "text-base transition-colors",
                          location.pathname === subItem.href 
                            ? "text-forestt-primary font-medium" 
                            : "text-foreground hover:text-forestt-primary"
                        )}
                      >
                        {subItem.name}
                      </Link>
                      
                      {subItem.hasSubDropdown && (
                        <button
                          onClick={(e) => toggleSubMenu(subItem.name, e)}
                          className="p-1 rounded-md"
                          aria-expanded={openSubMenus.includes(subItem.name)}
                          aria-label={`Afficher le sous-menu de ${subItem.name}`}
                        >
                          {openSubMenus.includes(subItem.name) ? (
                            <ChevronUp className="h-4 w-4" />
                          ) : (
                            <ChevronDown className="h-4 w-4" />
                          )}
                        </button>
                      )}
                    </div>

                    {/* Sous-sous-menus avec indentation supplémentaire */}
                    {subItem.hasSubDropdown && openSubMenus.includes(subItem.name) && (
                      <div className="pl-4 mt-1 mb-2 space-y-2 border-l-2 border-forestt-primary/20">
                        {subItem.subDropdownItems.map((subSubItem) => (
                          <div key={subSubItem.name} className="flex flex-col">
                            <Link 
                              to={subSubItem.href}
                              className={cn(
                                "py-2 text-sm transition-colors",
                                location.pathname === subSubItem.href 
                                  ? "text-forestt-primary font-medium" 
                                  : "text-foreground hover:text-forestt-primary"
                              )}
                            >
                              {subSubItem.name}
                            </Link>
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                ))}
              </div>
            )}
          </div>
        ))}
      </nav>
    </div>
  );
};

export default MobileNavigation;
