
import { Search, Loader2 } from "lucide-react";
import { cn } from "@/lib/utils";
import { useState, useEffect } from "react";

interface NewsSearchProps {
  searchTerm: string;
  setSearchTerm: (value: string) => void;
  activeTab: string;
  setActiveTab: (value: string) => void;
  tabs: Array<{ id: string; label: string }>;
}

const NewsSearch = ({ searchTerm, setSearchTerm, activeTab, setActiveTab, tabs }: NewsSearchProps) => {
  const [isLoading, setIsLoading] = useState(false);

  // Simulate loading state for better UX
  useEffect(() => {
    if (searchTerm.trim() === "") {
      setIsLoading(false);
      return;
    }
    
    setIsLoading(true);
    const timer = setTimeout(() => {
      setIsLoading(false);
    }, 300);
    
    return () => clearTimeout(timer);
  }, [searchTerm]);

  return (
    <div className="max-w-2xl mx-auto mt-8">
      <div className="relative">
        {isLoading ? (
          <Loader2 className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground animate-spin" aria-hidden="true" />
        ) : (
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-5 w-5 text-muted-foreground" aria-hidden="true" />
        )}
        <input
          type="text"
          placeholder="Rechercher des actualités..."
          className="w-full py-3 pl-10 pr-4 rounded-lg border border-ecofor-medium-blue/40 focus:outline-none focus:ring-2 focus:ring-ecofor-dark-green focus:border-transparent"
          value={searchTerm}
          onChange={(e) => setSearchTerm(e.target.value)}
          aria-label="Rechercher des actualités"
        />
      </div>
      
      <div className="flex justify-center mt-6">
        <div className="inline-flex p-1 bg-white rounded-lg shadow-sm" role="tablist">
          {tabs.map((tab) => (
            <button
              key={tab.id}
              role="tab"
              aria-selected={activeTab === tab.id}
              aria-controls={`${tab.id}-tab`}
              id={`${tab.id}-button`}
              className={cn(
                "px-4 py-2 text-sm font-medium rounded-md transition-colors",
                activeTab === tab.id
                  ? "bg-ecofor-dark-green text-white"
                  : "text-foreground hover:bg-ecofor-light-blue"
              )}
              onClick={() => setActiveTab(tab.id)}
            >
              {tab.label}
            </button>
          ))}
        </div>
      </div>
    </div>
  );
};

export default NewsSearch;
