
import { useState, useEffect, useRef } from "react";
import { useNavigate } from "react-router-dom";
import { SearchResult } from "@/types/search";
import { searchGlobal, getTypeLabel } from "@/services/searchService";
import { Search, X, FileText, Calendar, ArrowRight, Loader2 } from "lucide-react";
import { Dialog, DialogContent, DialogTitle } from "@/components/ui/dialog";
import { Command, CommandInput, CommandList, CommandEmpty, CommandGroup, CommandItem } from "@/components/ui/command";
import { VisuallyHidden } from "@/components/ui/visually-hidden";
import { cn } from "@/lib/utils";

interface SearchDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
}

const SearchDialog = ({ open, onOpenChange }: SearchDialogProps) => {
  const [searchQuery, setSearchQuery] = useState("");
  const [results, setResults] = useState<SearchResult[]>([]);
  const [isSearching, setIsSearching] = useState(false);
  const inputRef = useRef<HTMLInputElement>(null);
  const navigate = useNavigate();

  // Focus input when dialog opens
  useEffect(() => {
    if (open && inputRef.current) {
      setTimeout(() => {
        inputRef.current?.focus();
      }, 100);
    }
  }, [open]);

  // Debounced search
  useEffect(() => {
    if (!searchQuery.trim()) {
      setResults([]);
      return;
    }

    const timer = setTimeout(() => {
      setIsSearching(true);
      try {
        const searchResults = searchGlobal(searchQuery);
        setResults(searchResults);
      } catch (error) {
        console.error("Search error:", error);
      } finally {
        setIsSearching(false);
      }
    }, 300);

    return () => clearTimeout(timer);
  }, [searchQuery]);

  const handleSelect = (url: string) => {
    navigate(url);
    onOpenChange(false);
    setSearchQuery("");
  };

  const getTypeIcon = (type: string) => {
    switch (type) {
      case 'news':
        return <Calendar className="h-4 w-4 text-ecofor-turquoise" />;
      case 'resources':
        return <FileText className="h-4 w-4 text-ecofor-dark-green" />;
      default:
        return <FileText className="h-4 w-4" />;
    }
  };

  return (
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent 
        className="p-0 gap-0 max-w-3xl" 
        aria-describedby="search-description"
        onEscapeKeyDown={() => onOpenChange(false)}
        onInteractOutside={() => onOpenChange(false)}
      >
        <DialogTitle className="sr-only">
          <VisuallyHidden>Recherche sur le site</VisuallyHidden>
        </DialogTitle>
        <Command className="rounded-lg border shadow-md">
          <div className="flex items-center border-b px-3 w-full">
            {isSearching ? (
              <Loader2 className="mr-2 h-5 w-5 shrink-0 animate-spin opacity-50" />
            ) : (
              <Search className="mr-2 h-5 w-5 shrink-0 opacity-50" />
            )}
            <CommandInput
              ref={inputRef}
              value={searchQuery}
              onValueChange={setSearchQuery}
              placeholder="Rechercher sur tout le site..."
              className="flex h-14 w-full rounded-md bg-transparent py-3 text-base outline-none placeholder:text-muted-foreground"
              aria-label="Rechercher sur tout le site"
            />
            {searchQuery && (
              <button 
                onClick={() => setSearchQuery("")}
                className="text-muted-foreground hover:text-foreground p-1"
                aria-label="Effacer la recherche"
              >
                <X className="h-5 w-5" />
              </button>
            )}
          </div>
          <CommandList className="max-h-[60vh] overflow-y-auto">
            <CommandEmpty>
              {isSearching ? (
                <div className="flex flex-col items-center justify-center py-6">
                  <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
                  <p className="mt-2 text-sm text-muted-foreground">Recherche en cours...</p>
                </div>
              ) : (
                "Aucun résultat trouvé."
              )}
            </CommandEmpty>
            {results.length > 0 && (
              <CommandGroup>
                {results.map((result) => (
                  <CommandItem
                    key={`${result.type}-${result.id}`}
                    onSelect={() => handleSelect(result.url)}
                    className="flex items-start py-3 px-4 cursor-pointer"
                  >
                    <div className="flex flex-col space-y-1 flex-1">
                      <div className="flex items-center">
                        <span className="inline-flex items-center mr-2 px-2 py-1 rounded-full text-xs font-medium bg-ecofor-light-blue text-ecofor-dark-green">
                          {getTypeIcon(result.type)}
                          <span className="ml-1">{getTypeLabel(result.type)}</span>
                        </span>
                        {result.date && (
                          <span className="text-xs text-muted-foreground">
                            {result.date}
                          </span>
                        )}
                      </div>
                      <div className="font-medium">{result.title}</div>
                      {result.description && (
                        <p className="text-sm text-muted-foreground line-clamp-2">
                          {result.description}
                        </p>
                      )}
                    </div>
                    <ArrowRight className="h-4 w-4 ml-2 flex-shrink-0 mt-2 text-muted-foreground" />
                  </CommandItem>
                ))}
              </CommandGroup>
            )}
          </CommandList>
        </Command>
      </DialogContent>
    </Dialog>
  );
};

export default SearchDialog;
