
import React, { useEffect, useState } from 'react';
import { fetchWordPressData, parseWordPressContent, wordpressConfig } from '../../wordpress-config';
import { Skeleton } from '../ui/skeleton';
import EditorAdapter from '../../wordpress/EditorAdapter';
import { toast } from '../ui/use-toast';
import { Button } from '../ui/button';
import { ArrowLeft, Bookmark, Calendar, Download, Eye, FileText, Tag, User } from 'lucide-react';
import { Link, useNavigate } from 'react-router-dom';
import { formatDistanceToNow } from 'date-fns';
import { fr } from 'date-fns/locale';

interface WordPressIntegrationProps {
  contentType: string;
  slug?: string;
  params?: Record<string, string>;
  fallbackComponent?: React.ReactNode;
  showMeta?: boolean;
  showFeaturedImage?: boolean;
  showTaxonomies?: boolean;
  showAuthor?: boolean;
  showDate?: boolean;
  showBackButton?: boolean;
}

const WordPressIntegration: React.FC<WordPressIntegrationProps> = ({
  contentType,
  slug,
  params = {},
  fallbackComponent,
  showMeta = true,
  showFeaturedImage = true,
  showTaxonomies = true,
  showAuthor = true,
  showDate = true,
  showBackButton = false
}) => {
  const [content, setContent] = useState<any>(null);
  const [loading, setLoading] = useState<boolean>(true);
  const [error, setError] = useState<boolean>(false);
  const navigate = useNavigate();

  useEffect(() => {
    const fetchContent = async () => {
      setLoading(true);
      try {
        let data;
        if (slug) {
          data = await fetchWordPressData(contentType, { slug, ...params, _embed: true });
        } else {
          data = await fetchWordPressData(contentType, { ...params, _embed: true });
        }
        
        if (Array.isArray(data) && data.length > 0) {
          setContent(data[0]);
        } else if (data && !Array.isArray(data)) {
          setContent(data);
        } else {
          setError(true);
          toast({
            title: "Contenu non trouvé",
            description: "Impossible de charger le contenu depuis WordPress",
            variant: "destructive"
          });
        }
      } catch (err) {
        console.error("Erreur lors du chargement du contenu WordPress:", err);
        setError(true);
        toast({
          title: "Erreur de chargement",
          description: "Problème de connexion à l'API WordPress",
          variant: "destructive"
        });
      } finally {
        setLoading(false);
      }
    };

    fetchContent();
  }, [contentType, slug, JSON.stringify(params)]);

  if (loading) {
    return (
      <div className="space-y-4 p-4">
        <Skeleton className="h-12 w-3/4" />
        <Skeleton className="h-4 w-full" />
        <Skeleton className="h-4 w-full" />
        <Skeleton className="h-4 w-3/4" />
        <Skeleton className="h-64 w-full" />
      </div>
    );
  }

  if (error || !content) {
    return fallbackComponent || (
      <div className="p-4 text-center">
        <h2 className="text-lg font-semibold mb-2">Contenu non disponible</h2>
        <p>Le contenu demandé n'a pas pu être chargé depuis WordPress.</p>
      </div>
    );
  }

  // Récupérer les métadonnées
  const title = content.title?.rendered || content.title;
  const featuredImage = content.featured_media && content._embedded?.['wp:featuredmedia']?.[0]?.source_url;
  const author = content._embedded?.['author']?.[0]?.name || 'GIP Ecofor';
  const date = content.date ? new Date(content.date) : new Date();
  const formattedDate = formatDistanceToNow(date, { addSuffix: true, locale: fr });
  
  // Récupérer les taxonomies
  const categories = content._embedded?.['wp:term']?.[0] || [];
  const tags = content._embedded?.['wp:term']?.[1] || [];
  
  // Détecter le type d'éditeur utilisé
  const editorType = content.meta?.editor_type || 'gutenberg';
  
  // Obtenir les métadonnées ACF
  const acfData = content.acf || {};
  
  // Obtenir le lien de téléchargement si disponible
  const downloadLink = acfData.file_download || acfData.download_link || null;
  
  return (
    <article className="wordpress-content-wrapper">
      {showBackButton && (
        <Button 
          variant="ghost" 
          onClick={() => navigate(-1)} 
          className="mb-4 flex items-center gap-2"
        >
          <ArrowLeft size={16} />
          Retour
        </Button>
      )}
      
      <h1 className="text-2xl md:text-3xl font-bold mb-6 text-forestt-primary">{title}</h1>
      
      {showMeta && (
        <div className="flex flex-wrap items-center gap-4 text-sm text-muted-foreground mb-6">
          {showDate && (
            <div className="flex items-center gap-1">
              <Calendar size={16} />
              <span>{formattedDate}</span>
            </div>
          )}
          
          {showAuthor && (
            <div className="flex items-center gap-1">
              <User size={16} />
              <span>{author}</span>
            </div>
          )}
          
          {acfData.views && (
            <div className="flex items-center gap-1">
              <Eye size={16} />
              <span>{acfData.views} vues</span>
            </div>
          )}
          
          {acfData.document_type && (
            <div className="flex items-center gap-1">
              <FileText size={16} />
              <span>{acfData.document_type}</span>
            </div>
          )}
        </div>
      )}
      
      {showTaxonomies && (categories.length > 0 || tags.length > 0) && (
        <div className="flex flex-wrap gap-2 mb-6">
          {categories.map((term: any) => (
            <div 
              key={term.id} 
              className="inline-flex items-center gap-1 text-xs bg-forestt-primary/20 text-forestt-primary px-2 py-1 rounded-full"
            >
              <Bookmark size={12} />
              {term.name}
            </div>
          ))}
          
          {tags.map((term: any) => (
            <div 
              key={term.id} 
              className="inline-flex items-center gap-1 text-xs bg-forestt-secondary/20 text-forestt-secondary px-2 py-1 rounded-full"
            >
              <Tag size={12} />
              {term.name}
            </div>
          ))}
        </div>
      )}
      
      {showFeaturedImage && featuredImage && (
        <div className="featured-image mb-6">
          <img 
            src={featuredImage}
            alt={content._embedded['wp:featuredmedia'][0].alt_text || title} 
            className="w-full h-auto rounded-lg"
          />
          {content._embedded['wp:featuredmedia'][0].caption?.rendered && (
            <p 
              className="text-sm text-center text-muted-foreground mt-2"
              dangerouslySetInnerHTML={{ 
                __html: content._embedded['wp:featuredmedia'][0].caption.rendered
              }}
            />
          )}
        </div>
      )}
      
      <EditorAdapter 
        content={content.content?.rendered || ''} 
        editor={editorType}
      />
      
      {downloadLink && (
        <div className="mt-8 flex justify-center">
          <Button 
            variant="default" 
            className="flex items-center gap-2 bg-forestt-primary hover:bg-forestt-primary/90"
            asChild
          >
            <a href={downloadLink} target="_blank" rel="noopener noreferrer">
              <Download size={16} />
              Télécharger le document
            </a>
          </Button>
        </div>
      )}
    </article>
  );
};

export default WordPressIntegration;
