
import React, { useState } from 'react';
import { Helmet } from "react-helmet-async";
import PageLayout from "@/components/layout/PageLayout";
import ResourceTabs from "@/components/resources/ResourceTabs";
import { presentationsData } from '@/data/presentations';
import PresentationsList from '@/components/resources/PresentationsList';
import FilterTags from '@/components/resources/FilterTags';
import EmptyState from '@/components/resources/EmptyState';

const Presentations = () => {
  const allPresentations = presentationsData;
  const [filteredPresentations, setFilteredPresentations] = useState(allPresentations);
  const [activeFilter, setActiveFilter] = useState<string | null>(null);
  
  // Définir les filtres disponibles (par exemple par année)
  const filters = [
    { id: "2023", label: "2023" },
    { id: "2022", label: "2022" },
    { id: "all", label: "Toutes les années" },
  ];

  const handleFilterChange = (filterId: string) => {
    if (filterId === "all" || filterId === activeFilter) {
      setActiveFilter(null);
      setFilteredPresentations(allPresentations);
    } else {
      setActiveFilter(filterId);
      setFilteredPresentations(
        allPresentations.filter(presentation => 
          presentation.date.includes(filterId)
        )
      );
    }
  };

  const clearFilters = () => {
    setActiveFilter(null);
    setFilteredPresentations(allPresentations);
  };

  return (
    <PageLayout>
      <Helmet>
        <title>Présentations - FORESTT | Résilience des forêts</title>
        <meta name="description" content="Explorez les présentations scientifiques du programme FORESTT sur la résilience des forêts." />
      </Helmet>
      
      <section className="py-16">
        <div className="container max-w-5xl mx-auto">
          <div className="max-w-3xl mx-auto text-center mb-10">
            <h1 className="mb-6 text-4xl font-bold text-forestt-primary">Présentations</h1>
            <p className="text-lg mb-6 text-gray-600">
              Retrouvez les présentations réalisées par les chercheurs du programme FORESTT lors de conférences et d'événements scientifiques.
            </p>
          </div>
          
          <ResourceTabs />

          <div className="mb-8">
            <FilterTags 
              filters={filters} 
              activeFilter={activeFilter} 
              onChange={handleFilterChange} 
              onClear={clearFilters}
            />
          </div>
          
          {filteredPresentations.length > 0 ? (
            <PresentationsList presentations={filteredPresentations} />
          ) : (
            <EmptyState 
              message="Aucune présentation trouvée pour ce filtre." 
              buttonText="Voir toutes les présentations" 
              onButtonClick={clearFilters}
            />
          )}
        </div>
      </section>
    </PageLayout>
  );
};

export default Presentations;
