
/**
 * Mappage entre les templates WordPress et les composants React
 * Ce fichier permet de définir comment les templates WordPress sont
 * rendus par les composants React correspondants.
 */

import Presentation from "@/pages/about/Presentation";
import WorkPackages from "@/pages/about/WorkPackages";
import Partners from "@/pages/about/Partners";
import Chercheurs from "@/pages/about/Chercheurs";
import Deliverables from "@/pages/resources/Deliverables";
import Publications from "@/pages/resources/Publications";
import Videos from "@/pages/resources/Videos";
import Presentations from "@/pages/resources/Presentations";
import Data from "@/pages/resources/Data";
import DoctoralSchools from "@/pages/activites/DoctoralSchools";
import Colloques from "@/pages/activites/Colloques";
import Residences from "@/pages/activites/Residences";
import News from "@/pages/News";
import Contact from "@/pages/Contact";
import Index from "@/pages/Index";
import About from "@/pages/About";
import Resources from "@/pages/Resources";
import Activites from "@/pages/Activites";
import NotFound from "@/pages/NotFound";
import AdminAccess from "@/pages/AdminAccess";

// Mappage des templates WordPress vers les composants React
export const templateMapping = {
  // Templates de page par défaut
  'default': Index,
  'template-about': About,
  'template-resources': Resources,
  'template-activites': Activites,
  'template-news': News,
  'template-contact': Contact,
  
  // Templates de page personnalisés
  'template-presentation': Presentation,
  'template-work-packages': WorkPackages,
  'template-partners': Partners,
  'template-collaborators': Chercheurs,
  'template-deliverables': Deliverables,
  'template-publications': Publications,
  'template-videos': Videos,
  'template-presentations': Presentations,
  'template-data': Data,
  'template-doctoral-schools': DoctoralSchools,
  'template-colloques': Colloques,
  'template-residences': Residences,
  
  // Page admin cachée
  'template-admin': AdminAccess,
  
  // Fallback pour les pages non trouvées
  '404': NotFound
};

// Mappage des slugs WordPress vers les routes React
export const slugToRouteMapping = {
  // Pages principales
  '': '/',
  'about': '/about',
  'resources': '/resources',
  'activites': '/activites',
  'news': '/news',
  'contact': '/contact',
  
  // Sous-pages À propos
  'presentation': '/about/presentation',
  'work-packages': '/about/work-packages',
  'partners': '/about/partners',
  'collaborators': '/about/chercheurs',
  
  // Sous-pages Ressources
  'deliverables': '/resources/deliverables',
  'publications': '/resources/publications',
  'presentations': '/resources/presentations',
  'data': '/resources/data',
  'videos': '/resources/videos',
  
  // Sous-pages Activites
  'doctoral-schools': '/activites/doctoral-schools',
  'colloques': '/activites/colloques',
  'residences': '/activites/residences',
  
  // Page admin cachée
  'admin-forestt-hub-access': '/admin-forestt-hub-access'
};

// Mappage inverse des routes React vers les templates WordPress
export const routeToTemplateMapping = {
  '/': 'default',
  '/about': 'template-about',
  '/about/presentation': 'template-presentation',
  '/about/work-packages': 'template-work-packages',
  '/about/partners': 'template-partners',
  '/about/chercheurs': 'template-collaborators',
  '/about/collaborators': 'template-collaborators',
  '/resources': 'template-resources',
  '/resources/deliverables': 'template-deliverables',
  '/resources/publications': 'template-publications',
  '/resources/presentations': 'template-presentations',
  '/resources/data': 'template-data',
  '/resources/videos': 'template-videos',
  '/activites': 'template-activites',
  '/activites/doctoral-schools': 'template-doctoral-schools',
  '/activites/colloques': 'template-colloques',
  '/activites/residences': 'template-residences',
  '/news': 'template-news',
  '/contact': 'template-contact',
  '/admin-forestt-hub-access': 'template-admin'
};

// Fonction pour déterminer le composant React à utiliser en fonction du template WordPress
export const getComponentForTemplate = (templateName: string) => {
  return templateMapping[templateName] || templateMapping['default'];
};

// Fonction pour convertir un slug WordPress en route React
export const convertSlugToRoute = (slug: string) => {
  return slugToRouteMapping[slug] || `/${slug}`;
};

// Fonction pour convertir une route React en template WordPress
export const convertRouteToTemplate = (route: string) => {
  return routeToTemplateMapping[route] || 'default';
};

// Fonction pour déterminer si une route est une sous-route
export const isSubRoute = (route: string) => {
  return route.split('/').length > 2;
};

// Fonction pour obtenir la route parente d'une sous-route
export const getParentRoute = (route: string) => {
  const parts = route.split('/');
  if (parts.length <= 2) return route;
  return `/${parts[1]}`;
};
